import AutoComplete from "@tarekraafat/autocomplete.js/dist/autoComplete";
/**
 * The list of cities
 * @type {[] | undefined}
 */
let cities = undefined;
let loading = false;
/**
 * CityAutocomplete component
 */
class CityAutocomplete {

    /**
     * The input element
     * @type {string}
     */
    ajaxUrl = '';

    /**
     * The autocomplete instance
     * @type {undefined | AutoComplete}
     */
    autocomplete = undefined;

    /**
     * The input text
     * @type {string}
     */
    currentText = '';

    /**
     * Initialized?
     * @type {boolean}
     */
    initialized = false;

    /**
     * @constructor
     * @param element {HTMLElement} The button element
     */
    constructor(element) {
        this.element = element;
        // Remove the native browser error message
        this.ajaxUrl = window.a42?.ajaxurl;
        this.validationElement = this.element;
        this.init();
    }

    init() {
        this.validationElement.getValidity = this.getValidity.bind(this);
        this.element.addEventListener('focus', async () => {
            if (cities == null && !loading && !this.initialized) {
                loading = true;
                const data = await this._fetchData();
                if (data.success === true) {
                    cities = data.data;
                    this.initAutoComplete();
                    this.element.focus();

                } else {
                    console.error('Error fetching cities data');
                    loading = false;
                }
            } else if (!this.initialized) {
                this.initAutoComplete()
                this.element.focus();
            } else {
                this.element.focus();
                this.autocomplete.open();
            }
        });

        // Focus out
        this.element.addEventListener('focusout', () => {
            if (!this.initialized) {
                return;
            }
            this.element.dispatchEvent(new Event('validate'));
        });
    }

    getValidity() {

        const isFieldRequired = this.element.getAttribute('required') != null;

        let isValid = true;

        if (!isFieldRequired && this.element.value === '') {
            return true;
        }

        if (isFieldRequired) {
            isValid = this.element.value !== '';
            if (!isValid) {
                this.validationElement.error = 'Veuillez sélectionner une ville';
            } else {
                this.validationElement.error = '';
            }
        }

        // Check if the value is in the list
        if (isValid) {
            const value = this.element.value;
            if (!cities.includes(value)) {
                isValid = false;
                this.validationElement.error = 'Veuillez sélectionner une ville parmi la liste';
            } else {
                this.validationElement.error = '';
            }
        }

        return isValid;
    }

    /**
     * Format the string to ignore spaces and special characters
     * @param str
     * @returns {string}
     */
    _formatString(str) {
        // Replace all special characters and spaces êàèé -> eae
        str = str.normalize('NFD').replace(/[\u0300-\u036f]/g, "");
        return str.toLowerCase().replace(/[^a-zA-Z0-9]/g, '');
    }

    /**
     * Matches the query with the record
     *
     * The record is formatted as : "12345, City Name"
     * The query will match if the postal code is found before or after the city name
     * @param query {string} The query string
     * @param record {string} The record string
     * @returns {boolean}
     */
    _matches(query, record) {
        const recordPostCode = this._formatString(record.split(',')[0]);
        const recordCity = this._formatString(record.split(',')[1]);
        const q = this._formatString(query);
        // Get all characters from the query
        const qPostCodes = this._formatString(q.match(/\d/g)?.join('') ?? '');
        const qCities = this._formatString(q.match(/[a-zA-Z]/g)?.join('') ?? '');
        // Get all characters from the record
        if (qPostCodes != null && qCities != null) {
            // Check if all characters from the query are in the record
            return recordPostCode.includes(qPostCodes) && recordCity.includes(qCities);
        } else if (qPostCodes != null) {
            return recordPostCode.includes(qPostCodes);
        } else if (qCities != null) {
            return recordCity.includes(qCities);
        } else {
            return false;
        }
    }

    /**
     * Search engine
     * @param query
     * @return {Promise<T[]>}
     */
    async searchEngine(query) {
        const q = this._formatString(query);

        const filteredCities = cities.filter(city => {
            return this._matches(query, city);
        });

        return filteredCities.sort((a, b) => {
            const aIndex = this._formatString(a).indexOf(q);
            const bIndex = this._formatString(b).indexOf(q);
            if (aIndex === bIndex) {
                return a.localeCompare(b);
            }
            return aIndex - bIndex;
        });

    }

    /**
     * Initialize the Autocomplete
     */
    initAutoComplete() {
        this.autocomplete = new AutoComplete({
            data: {
                src: query => this.searchEngine(query),
                cache: false,
            },
            /**
             * Render the results
             *
             * Strategy:
             * 1. 2 Firsts letters must match
             * 2. Need at least 2 letters
             * 3. Case-insensitive
             *
             * @param query
             * @param record
             * @return {*}
             */
            searchEngine: (query, record) => {
                //console.log('record:', record)
                return record;
            },
            resultsList: {
                maxResults: 15,
                class: "a42-autocomplete-results",
            },
            threshold: 2,
            selector: () => {
                return this.element;
            },
            events: {
                input: {
                    selection: (event) => {
                        const selection = event.detail.selection.value;
                        this.autocomplete.input.value = selection;
                        this.element.dispatchEvent(new Event('validate'));
                    },
                    results: (event) => {

                        //console.log('results:', event)

                        if (event.detail.results.length === 0) {
                            //this.errorBox.innerText = "Aucune ville ne correspond à votre recherche";
                        }
                    },
                    close: (event) => {
                        //this.errorBox.style.display = "block";
                        //this.element.dispatchEvent(new Event('validate'));
                    },
                    open: (event) => {
                        //this.errorBox.style.display = "none";
                    },
                },
            },
        });
        this.initialized = true;
    }

    /**
     * Fetch data from the server
     * @returns {Promise<unknown>}
     */
    _fetchData() {
        return new Promise((resolve, reject) => {
            const formData = new FormData();
            formData.append('action', 'a42_city_autocomplete');

            fetch(this.ajaxUrl, {
                method: 'POST',
                body: formData,
            }).then(response => response.json())
                .then(body => {
                    resolve(body);
                }
            );
        });
    }
}


document.querySelectorAll("input[type='a42-city-autocomplete']").forEach(element => {
    new CityAutocomplete(element);
});