<?php
namespace Elementor42\Widgets;

use Elementor\Element_Base;

if ( ! defined( 'ABSPATH' ) ) {
  exit; // Exit if accessed directly.
}

/**
 * Class OptionFlexBasis
 *
 * @package Elementor42\Widgets
 */
class OptionFlexBasis {
  public function __construct() {
    // Add into layout options
    add_action('elementor/element/after_section_end', [$this, 'register_controls'], 10, 3);

    // Add class to wrapper
    add_action('elementor/frontend/before_render', [$this, 'add_class_to_wrapper'], 10, 1);


    // Add class to containers if they have flex-direction: row and flex-wrap: wrap
    add_action('elementor/frontend/container/before_render', [$this, 'add_class_to_container'], 10, 1);
  }

  /**
   * Add class to container if flex direction is row and flex wrap is wrap
   * @param $element Element_Base
   * @return void
   */
  public function add_class_to_container(Element_Base $element): void {
    $settings = $element->get_settings_for_display();
    if ( !empty($settings['flex_direction']) && $settings['flex_direction'] === 'row' && !empty($settings['flex_wrap']) && $settings['flex_wrap'] === 'wrap' ) {
      $element->add_render_attribute('_wrapper', 'class', 'flex42-container');
    }
  }

  /**
   * Register controls
   *
   * @param $element
   * @param $section_id
   * @param $args
   * @return void
   */
  public function register_controls($element, $section_id, $args): void {

    $allowed_elements = ['container', 'text-editor'];
    $allowed_sections = ['section_custom_css', 'section_drop_cap'];

    if ( /*in_array($element->get_name(), $allowed_elements) &&*/ in_array($section_id, $allowed_sections) ) {
      $element->start_controls_section(
        'section_option_flex_basis',
        [
          'label' => __('Flex42', ELEMENTOR42_SLUG),
          'tab' => \Elementor\Controls_Manager::TAB_LAYOUT,
        ]
      );
      $this->add_flex_basis_controls($element);

      if ($element->get_name() === 'container') {
        $this->add_flex_container_controls($element);
      }

      $element->end_controls_section();
    }

  }

  /**
   * Add class to wrapper if flex42 is active
   * @param $element Element_Base
   * @return void
   */
  public function add_class_to_wrapper(Element_Base $element): void {
    $settings = $element->get_settings_for_display();
    if ( !empty($settings['option_flex_basis_active']) && $settings['option_flex_basis_active'] === 'yes' ) {
      $element->add_render_attribute('_wrapper', 'class', 'flex42');
    }
  }

  /**
   * @param $element
   * @return void
   */
  private function add_flex_basis_controls($element): void {


    $element->add_control(
      'option_flex_basis_active',
      [
        'label' => __('Custom Flex Basis', ELEMENTOR42_SLUG),
        'type' => \Elementor\Controls_Manager::SWITCHER,
        'label_on' => __('Yes', ELEMENTOR42_SLUG),
        'label_off' => __('No', ELEMENTOR42_SLUG),
        'return_value' => 'yes',
        'default' => 'no',
      ]
    );

    $element->add_responsive_control(
      'option_flex_basis',
      [
        'label' => __('Flex Basis', ELEMENTOR42_SLUG),
        'type' => \Elementor\Controls_Manager::SELECT,
        'options' => [
          'auto' => __('Auto', ELEMENTOR42_SLUG),
          'content' => __('Content', ELEMENTOR42_SLUG),
          'fit-content' => __('Fit Content', ELEMENTOR42_SLUG),
          'max-content' => __('Max Content', ELEMENTOR42_SLUG),
          'min-content' => __('Min Content', ELEMENTOR42_SLUG),
          'inherit' => __('Inherit', ELEMENTOR42_SLUG),
          'initial' => __('Initial', ELEMENTOR42_SLUG),
          'unset' => __('Unset', ELEMENTOR42_SLUG),
          'manual' => __('Manual', ELEMENTOR42_SLUG),
        ],
        'condition' => [
          'option_flex_basis_active' => 'yes',
        ],
        'default' => 'manual',

        'selectors' => [
          '{{WRAPPER}}' => 'flex-basis: {{VALUE}};',
        ],
      ]
    );

    $element->add_responsive_control(
      'option_flex_basis_value',
      [
        'label' => __('Flex Basis Value', ELEMENTOR42_SLUG),
        'type' => \Elementor\Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'vw', 'vh', 'em', 'rem'],
        'range' => [
          'px' => [
            'min' => 0,
            'max' => 1000,
          ],
        ],
        'default' => [
          'unit' => 'px',
          'size' => 300,
        ],
        'condition' => [
          'option_flex_basis' => 'manual',
          'option_flex_basis_active' => 'yes',
        ],
        'selectors' => [
          '{{WRAPPER}}' => 'flex-basis: {{SIZE}}{{UNIT}};',
        ],
      ]
    );


  }

  /**
   * @param $element
   * @return void
   */
  private function add_flex_container_controls($element): void {

    // Gap between items (x-axis)
    /*$element->add_responsive_control(
      'option_flex42_gap_x',
      [
        'label' => __('Gap X', ELEMENTOR42_SLUG),
        'type' => \Elementor\Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'vw', 'vh', 'em', 'rem'],
        'range' => [
          'px' => [
            'min' => 0,
            'max' => 1000,
          ],
        ],
        'default' => [
          'unit' => 'px',
          'size' => 0,
        ],
        'selectors' => [
          '{{WRAPPER}}' => 'gap: {{SIZE}}{{UNIT}} 0;',
        ],
      ]
    );*/
  }
}