class Pagination42 {

    /**
     * Element
     * @type {Element}
     */
    element;

    /**
     * @constructor
     * @param element {Element}
     * @param loop42 {Loop42}
     */
    constructor(element, loop42) {
        this.element = element;
        this.loop42 = loop42;
        if (this.element == null) return;
        this.element.Pagination42 = this;
        this.init();
    }

    init() {
        this.listenEvents();
    }

    listenEvents() {
        // Listen to page numbers
        const pageNumElements =
            this.element.querySelectorAll('.page-numbers')
        pageNumElements.forEach(pageNumElement => {
            pageNumElement.addEventListener('click', (event) => {
                event.preventDefault();
                this.managePageNum(pageNumElement);
            });
        });
    }

    /**
     * Manage page number click
     * @param pageNumElement
     */
    managePageNum(pageNumElement) {
        const href = pageNumElement.getAttribute('href');
        if (href == null) return;
        this.loop42.queryPage(this.getPageNum(href));
    }

    /**
     * Get page number from url
     * @returns {number}
     */
    getPageNum(url) {
        const matches = url.match(/\/page\/(\d+)/);
        if (matches == null) return 1;
        return parseInt(matches[1]);
    }
}
class Loop42 {

    /**
     * Element Widget
     * @type {Element}
     */
    element;

    /**
     * Element Loop container
     * @type {Element}
     */
    elementLoopContainer;

    /**
     * Elementor Pagination container
     * @type {Element | undefined}
     */
    elementPaginationContainer;

    /**
     * Ajax URL
     * @type {string}
     */
    ajaxUrl;

    /**
     * Scroll top offset
     * @type {{
     *     desktop: number,
     *     tablet: number,
     *     mobile: number
     * }}
     */
    scrollTopOffset = {
        desktop: 0,
        tablet: 0,
        mobile: 0
    };

    /**
     * Document ID
     * @type {string}
     */
    documentId;

    /**
     * Document dataID
     * @param {string}
     */
    dataId;

    /**
     * Current args
     * @type {object}
     */
    currentPage = 1;

    /**
     * Current page url
     * @param {string}
     */
    currentUrl = '';

    /**
     * Current filters
     */
    currentArgs = {};

    /**
     * Pagination42
     * @type {Pagination42 | undefined}
     */
    pagination;

    /**
     * Cached data
     * @typedef {{
 *          query: {string},
 *          posts: {string},
 *          pagination: {string}
     * }} CachedData
     * @type CachedData[]
     */
    cachedData = [];

    /**
     * @constructor
     * @param element
     */
    constructor(element) {
        this.element = element;
        this.ajaxUrl = window.a42?.ajaxurl;

        if (this.element == null || this.ajaxUrl == null) return;
        this.elementLoopContainer = this.element.querySelector('#a42-loop');
        this.elementPaginationContainer = this.element.querySelector('.a42-loop42--pagination');
        this.element.Loop42 = this;
        this.documentId = this.elementLoopContainer.getAttribute('data-doc-id');
        this.scrollTopOffset.desktop = parseInt(this.elementLoopContainer.getAttribute('data-scroll-offset'));
        this.scrollTopOffset.tablet = parseInt(this.elementLoopContainer.getAttribute('data-scroll-offset-tablet'));
        this.scrollTopOffset.mobile = parseInt(this.elementLoopContainer.getAttribute('data-scroll-offset-mobile'));
        this.dataId = this.element.getAttribute('data-id');

        this.currentUrl = this._getUrl();
        this.init();
    }

    init() {
        this.attachPagination();
    }

    attachPagination() {
        if (this.elementPaginationContainer == null) return;
        const pagination = this.elementPaginationContainer.querySelector('.a42-loop42--pagination__nav');
        if (pagination == null) return;
        this.pagination = new Pagination42(pagination, this);

    }

    _getUrl() {
        const url = window.location.href;
        // Remove "/page/2" from url
        return url.replace(/\/page\/\d+/, '');
    }

    /**
     * Scroll to top of the loop container
     */
    scrollToTop() {
        const currentBreakpoint = document.querySelector('body')
            .getAttribute('data-elementor-device-mode') ?? 'desktop';
        window.scrollTo({
            top: this.elementLoopContainer.getBoundingClientRect().top + window.scrollY
                - this.scrollTopOffset[currentBreakpoint],
            behavior: 'smooth'
        });
    }

    /**
     * @method query posts
     * @param args {object}
     * @param args.page {number}
     * @param args.filters {number[]}
     * @param args.orderby {string}
     * @public
     */
    async query(args) {
        args.pagination_page = args.pagination_page ?? this.currentPage;
        args.page_url = args.page_url ?? this.currentUrl;
        this.currentArgs = args;

        // Check if cached
        const cached = this.cachedData.find(cached => cached.query === JSON.stringify(args));
        if (cached != null) {
            this._disappearItems();
            setTimeout(() => {
                this._renderPosts(cached.posts);
                this._renderPagination(cached.pagination);
                this.attachPagination();
            }, 200);

        } else {
            const body = await this._fetchData(this.currentArgs);
            this._renderPosts(body.data.posts);
            this._renderPagination(body.data.pagination);
            this.attachPagination();
            this.cachedData.push({
                posts: body.data.posts,
                pagination: body.data.pagination,
                query: JSON.stringify(this.currentArgs)
            });
        }


    }

    /**
     * @method query page
     * @param page {number}
     */
    async queryPage(page) {
        this.currentArgs.pagination_page = page;
        this.scrollToTop();
        await this.query(this.currentArgs);
    }

    /**
     * fetch data from server (ajax)
     * @param args {object}
     * @param args.page {number}
     * @param args.filters {number[]}
     * @param args.orderby {string}
     * @returns {Promise<any>}
     * @private
     */
    _fetchData(args) {
        return new Promise((resolve, reject) => {
            const formData = new FormData();
            formData.append('dataId', this.dataId);
            formData.append('documentId', this.documentId);
            formData.append('action', 'a42_loop42');
            formData.append('query', JSON.stringify(args));


            this._disappearItems();
            fetch(this.ajaxUrl, {
                method: 'POST',
                body: formData,
            }).then(response => response.json())
                .then(body => {
                    resolve(body);
                });
        });
    }

    /**
     * render posts
     * @param posts {string}
     * @private
     */
    _renderPosts(posts) {
        this.elementLoopContainer.classList.remove('disappear');
        this.elementLoopContainer.innerHTML = posts;
    }

    /**
     * render pagination
     * @param pagination {string}
     * @private
     */
    _renderPagination(pagination) {
        if (this.elementPaginationContainer == null) return;
        this.elementPaginationContainer.innerHTML = pagination;
    }

    /**
     * Make items disappear (animation)
     * @private
     */
    _disappearItems() {
        this.elementLoopContainer.classList.add('disappear');
    }
}

document.querySelectorAll('.elementor-widget-a42-loop42').forEach(element => {
    new Loop42(element);
});