
class scrollSize {

    /**
     * Breakpoint.
     * @type {number}
     */
    breakpoint = 1024;

    /**
     * The element.
     * @type {HTMLElement}
     */
    element;

    /**
     * @type {number}
     * @description The offset of scroll to start changing the size to mini.
     */
    scrolledOffset = 0;

    /**
     * @type {number}
     * @description The offset of scroll to change transparent to solid.
     */
    scrolledOffsetSolid = 0;

    /**
     * @type {string | null}
     * @description The background-color of the elementor section.
     */
    backgroundColor = null;

    /**
     * Body Element.
     * @type {HTMLElement}
     */
    body = document.querySelector('body');

    /**
     * Screen is mobile?
     * @type {boolean}
     */
    isMobile = false;

    /**
     * @constructor
     * @param element {HTMLElement} Menu item element.
     */
    constructor(element) {
        this.element = element;
        this.scrolledOffset = parseInt(element.getAttribute('scrolled-offset'));
        this.scrolledOffsetSolid = parseInt(element.getAttribute('transparent-to-solid'));
        this.backgroundColor = element.getAttribute('solid-color');

        const button = this.element.querySelector('.elementor-widget-elementor-menu42-menu-button');
        if (!!button) {
            this.breakpoint = parseInt(button.dataset.breakpoint);
        }
        this.init();
    }

    /**
     * Initialize the class.
     * @returns {void}
     */
    init() {
        // Add event listeners
        this.addEventListeners();

        // Force first load
        this.onScroll();
    }

    /**
     * Add event listeners.
     * @returns {void}
     */
    addEventListeners() {
        window.addEventListener('scroll', this.onScroll.bind(this));
        window.addEventListener('resize', this.onResize.bind(this));
    }

    /**
     * On scroll event.
     * @returns {void}
     */
    onScroll() {
        window.pageYOffset > this.scrolledOffset ? this.setScrolledSize() : this.setUnscrolledSize();
        if (!!this.backgroundColor) {
            window.pageYOffset > this.scrolledOffsetSolid ? this.setSolid() : this.setTransparent();
        }

    }

    /**
     * On resize event.
     * @returns {void}
     */
    onResize() {
        const changed = this.isMobile !== (window.innerWidth <= this.breakpoint);
        this.isMobile = window.innerWidth <= this.breakpoint;
    }

    /**
     * Set the scrolled size.
     * @returns {void}
     */
    setScrolledSize() {
        this.element.classList.add('scrolled');
        this.element.classList.add('mini');
        this.body.classList.add('scrolled');
    }

    /**
     * Set the unscrolled size.
     * @returns {void}
     */
    setUnscrolledSize() {
        this.body.classList.remove('scrolled');
        this.element.classList.remove('scrolled');
        this.element.classList.remove('mini');
    }

    setSolid() {
        this.element.classList.add('solid');
        this.element.style.background = this.backgroundColor;
    }

    setTransparent() {
        this.element.style.background = 'transparent';
        this.element.classList.remove('solid');
    }

}

document.querySelectorAll('#main-header').forEach(element =>
    new scrollSize(element)
);