<?php

/**
 * Utils Controls class for Elementor
 *
 * @package Elementor_Menu42\Utils
 * @subpackage Elementor_Menu42\Utils
 * @author SAHEB-JAMII Soheil <anym@hotmail.fr>
 */

namespace ElementorMenu42\Utils;

use Elementor\Icons_Manager;
use Elementor\Utils;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Group_Control_Typography;
use ElementorPro\Plugin;

abstract class BaseWidget extends Widget_Base {

  /**
   * @param $name string The name of the control
   * @param $selector string|array The CSS selector
   * @param $default_color string
   * @param $default_font string
   * @param $label_before string The label before the control
   *
   * @return void
   */
    protected function add_quick_typography_controls_responsive(
      string $name,
      string|array $selector,
      string $default_color = '',
      string $default_font = '',
      string $label_before = ''
    ): void {
        $this->add_quick_typography_controls(
            $name,
            $selector,
            $default_color,
            $default_font,
            $label_before,
            true
        );
    }

  /**
   * @param $name string The name of the control
   * @param $selector string|array The CSS selector
   * @param $default_color string
   * @param $default_font string
   * @param $label_before string The label before the control
   * @param $responsive bool
   *
   * @return void
   */
  protected function add_quick_typography_controls(
    string $name,
    string|array $selector,
    string $default_color = '',
    string $default_font = '',
    string $label_before = '',
    bool $responsive = false
  ): void {
    if (!empty($label_before)) {
      $this->add_control(
        $name . '_heading',
        [
          'label' => $label_before,
          'type' => Controls_Manager::HEADING,
          'separator' => 'before',
        ]
      );
    }

    if (is_array($selector)) {
      $wrapped_selector = '';
      foreach ($selector as $key => $value) {
        $wrapped_selector .= '{{WRAPPER}} ' . $value;
        if ($key !== count($selector) - 1) {
          $wrapped_selector .= ', ';
        }
      }
    } else {
      $wrapped_selector = $wrapped_selector = '{{WRAPPER}} ' . $selector;
    }

    $global_color = [];
    if (!empty($default_color)) {
      $global_color['default'] = $default_color;
      $color_params = [
        'label' => __( 'Couleur', ELEMENTOR_MENU42_SLUG ),
        'type' => Controls_Manager::COLOR,
        'global' => $global_color,
        'selectors' => [
          $wrapped_selector => 'color: {{VALUE}}',
        ]
      ];
    } else {
      $color_params = [
        'label' => __( 'Couleur', ELEMENTOR_MENU42_SLUG ),
        'type' => Controls_Manager::COLOR,
        'selectors' => [
          $wrapped_selector => 'color: {{VALUE}}',
        ]
      ];
    }


    if ($responsive) $this->add_responsive_control($name . '_color', $color_params);
    else $this->add_control($name . '_color', $color_params);

    $global_typography = [];
    if (!empty($default_font)) {
      $global_typography['default'] = $default_font;
    }

    $this->add_group_control(
      Group_Control_Typography::get_type(),
      [
        'name' => $name . '_typography',
        'global' => $global_typography,
        'selector' => $wrapped_selector,
      ]
    );
  }

  /**
   * Render the icon
   *
   * @param array|null $icon array [ 'value' => 'fas fa-shopping-cart', 'library' => 'fa-solid' ]
   * @param $element_key string The key of the element
   *
   * @return void
   */
  protected function render_icon(array | null $icon = [], string $element_key = 'icon' ): void {
    if ( empty( $icon['value'] ) ) {
      return;
    }

    $this->add_render_attribute( $element_key, 'class', $icon );

    // If $element_key don't have $element_key class already, add it
    if ( !preg_match( '/\b' . $element_key . '\b/', $this->get_render_attribute_string( $element_key ) ) ) {
      $this->add_render_attribute( $element_key, 'class', 'a42-' . $element_key );
    }

    $this->add_render_attribute( 'icon', 'aria-hidden', 'true' );
    $migrated = isset( $settings['__fa4_migrated']['selected_icon'] );
    $is_new = empty( $icon ) && Icons_Manager::is_migration_allowed(); ?>
    <?php if ( $is_new || $migrated ) :
      Icons_Manager::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
    else : ?>
      <i <?php $this->print_render_attribute_string( $element_key ); ?>></i>
    <?php endif;
  }
}