<?php
namespace Elementor42\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

trait LoopFilter42RenderTrait {

	protected function render(): void {
		$settings = $this->get_settings_for_display();
		$items = $this->_getItems($settings);

		$this->_setAttributes($settings);
		$this->_printMainWidget($settings, $items);
	}

	private function _printMainWidget($settings, $items): void {
		echo strtr('<ul {attr}>', ['{attr}' => $this->get_render_attribute_string('wrapper')]);
		if ($settings['show_all_button'] === 'yes') {
			echo $this->_renderItem((object)[
				'name' => $settings['all_button_label'],
				'term_id' => -1,
				'count' => 0,
				'slug' => '',
			], 'active');
		}
		echo $this->_getHtmlItems($items, false, 0);
		echo "</ul>";
	}

	/**
	 * Set the attributes
	 * @param $settings
	 */
	private function _setAttributes($settings): void {
		$this->add_render_attribute('wrapper', [
			'class' => [
				'a42-loop-filter42',
				'items-wrapper',
			],
			'data-level' => 0,
		]);
		if ($settings['has_expandable_children'] === 'yes') {
			$this->add_render_attribute('wrapper', 'data-expandable', 'true');
		}
		if ($settings['allow_select_multiple'] === 'yes') {
			$this->add_render_attribute('wrapper', 'data-multiple', 'true');
		}
		if ($settings['has_scroll'] === 'yes') {
			$this->add_render_attribute('wrapper', 'data-scrollable', 'true');
		}
		$this->add_render_attribute('children_wrapper', [
			'class' => [
				'a42-loop-filter42--children',
				'items-wrapper',
			]
		]);

		if (!empty($settings['direction']) && $settings['direction'] === 'column') {
			$this->add_render_attribute('wrapper', 'class', 'flex-column');
			$this->add_render_attribute('children_wrapper', 'class', 'flex-column');
		} else {
			$this->add_render_attribute('wrapper', 'class', 'flex');
			$this->add_render_attribute('children_wrapper', 'class', 'flex');
		}
	}

	/**
	 * @param $items
	 * @param bool $is_children
	 * @param int $level
	 *
	 * @return string
	 */
	private function _getHtmlItems($items, bool $is_children = false, int $level = 0): string {
		$html = '';
		if ($is_children) {
			$html .= strtr('<ul {attr} data-level="{level}">',
				[
					'{attr}' => $this->get_render_attribute_string('children_wrapper'),
					'{level}' => $level,
				]);
		}
		foreach ($items as $item) {
			$html .= $this->_renderItem($item, '', intval($level));
		}
		if ($is_children) {
			$html .= "</ul>";
		}

		return $html;
	}

	/**
	 * Get Render item
	 *
	 * @param $item
	 * @param string $classes
	 * @param int $level
	 *
	 * @return string
	 */
	private function _renderItem($item, string $classes = '', int $level = 0): string {
		$icon = $this->get_settings_for_display('children_arrow_icon');
		return strtr(
			'<li class="a42-loop-filter42--item-wrapper flex-column" data-term-id="{id}" data-count="{count}">
					<div class="a42-loop-filter42--item item flex gap-10 align-center justify-between{classes}">
						<span class="a42-loop-filter42--item__label">
							{title}
						</span>
						{icon}
					</div>
					{children}
				</li>',
			[
				'{classes}' => !empty($classes) ? ' ' . $classes : '',
				'{title}' => $item->name,
				'{id}' => $item->term_id,
				'{count}' => $item->count,
				'{slug}' => $item->slug,
				'{children}' => !empty($item->children) ?
					$this->_getHtmlItems($item->children, true, intval($level) + 1) : '',
				'{icon}' => !empty($item->children) ?
					$this->get_render_icon($icon, 'children_arrow_icon') : '',
			]
		);
	}

	/**
	 * @param $settings
	 * Returns the items to display
	 * @example if taxonomy is on "category", return all the categories
	 */
	private function _getItems($settings): \WP_Error|array|int {
		$items = [];

		// IF taxonomy is on "category", return all the categories
		if ($settings['taxonomy'] === 'category') {
			//$items = get_categories();
			$items = get_terms([
				'taxonomy' => 'category',
				'exclude' => $settings['exclude_taxonomy_category'],
				'hide_empty' => false,
			]);

			$items = $this->_makeHierarchical($items, $settings['start_level']);
		}

		// IF taxonomy is on "tag", return all the tags
		if ($settings['taxonomy'] === 'tag') {
			$items = get_tags();
		}

		// IF taxonomy is on "product_cat", return all the product categories
		if ($settings['taxonomy'] === 'product_cat') {
			$items = get_terms([
				'taxonomy' => 'product_cat',
				'exclude' => $settings['exclude_taxonomy_product_cat'],
				'hide_empty' => false,
			]);

			$items = $this->_makeHierarchical($items, $settings['start_level']);
		}

		// IF taxonomy is on manual_category, return the manual selection
		if ($settings['taxonomy'] === 'manual_category') {
			$items = get_categories([
				'include' => $settings['filter_category_manual_selection'],
			]);
			$items = $this->_makeHierarchical($items, $settings['start_level']);
		}

		return $items;
	}

	/**
	 * Make hierarchical array of items from flat array items
	 *
	 * @param $flat
	 * @param int $parent
	 *
	 * @return array
	 */
	private function _makeHierarchical($flat, int $parent = 0): array {

		// We must find the first level items (if no items with parent = 0, start at 1)
		$current_parent = $this->_find_parent($flat, $parent);
		return $this->_getHierarchy($flat, $current_parent);
	}

	private function _find_parent($flat, $parent): int {
		foreach ($flat as $item) {
			if ($item->parent === $parent) {
				return $item->parent;
			}
		}
		if ($parent < 6) {
			return $this->_find_parent($flat, $parent + 1);
		} else {
			return 0;
		}
	}

	private function _getHierarchy($flat, $parent = 0): array {
		$hierarchical = [];
		foreach ($flat as $item) {
			if ($item->parent === $parent) {
				$children = $this->_getHierarchy($flat, $item->term_id);
				if ($children) {
					$item->children = $children;
				}
				$hierarchical[] = $item;
			}
		}
		return $hierarchical;
	}
}