<?php
namespace ElementorMenu42\Widgets;

use Elementor\Controls_Manager;
use ElementorMenu42\Utils\BaseWidget;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Class Menu Button
 *
 * @package ElementorMenu42\ElementorWidgets
 */
class MenuButton extends BaseWidget {

  public function get_name(): string {
    return 'elementor-menu42-menu-button';
  }

  public function get_title(): ?string {
    return __( '42 Menu Button', 'smatis' );
  }

  public function get_icon(): string {
    return 'eicon-menu-bar';
  }

  public function get_categories(): array {
    return [ 'agency42' ];
  }

  public function get_keywords(): array {
    return [ '42', 'menu', 'link', 'agence', 'button' ];
  }

  public function get_style_depends(): array {
    wp_register_style('a42-widget-menu-button', ELEMENTOR_MENU42_URL . 'assets/css/components/menu-button.css', [], ELEMENTOR_MENU42_VERSION);
    return ['a42-widget-menu-button'];
  }

  public function get_script_depends(): array {
    wp_register_script('a42-widget-menu-button', ELEMENTOR_MENU42_URL . 'assets/js/menu-button.js', [], ELEMENTOR_MENU42_VERSION, true);
    return ['a42-widget-menu-button'];
  }

  public function register_controls(): void {
    $this->start_controls_section(
      'section_content_menu',
      [
        'label' => __( 'Content', ELEMENTOR_MENU42_SLUG ),
        'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
      ]
    );

    // Add Breakpoint for menu into mobile (slider value)
    $this->add_control(
      'breakpoint',
      [
        'label' => __( 'Breakpoint', ELEMENTOR_MENU42_SLUG ),
        'type' => \Elementor\Controls_Manager::SLIDER,
        'size_units' => [ 'px' ],
        'range' => [
          'px' => [
            'min' => 0,
            'max' => 4000,
            'step' => 1,
          ],
        ],
        'default' => [
          'unit' => 'px',
          'size' => 1024,
        ],
      ]
    );

    $this->end_controls_section();

    $this->start_controls_section(
      'section_style_menu',
      [
        'label' => __( 'Style', ELEMENTOR_MENU42_SLUG ),
        'tab' => \Elementor\Controls_Manager::TAB_STYLE,
      ]
    );

    $this->add_control(
      'bar_color_label',
      [
        'label' => __( 'Bar Color', ELEMENTOR_MENU42_SLUG ),
        'type' => Controls_Manager::HEADING,
        'separator' => 'before',
      ]
    );

    $this->add_control(
      'bar_color',
      [
        'label' => __( 'Color', ELEMENTOR_MENU42_SLUG ),
        'type' => \Elementor\Controls_Manager::COLOR,
        'default' => '#000000',
        'selectors' => [
          '{{WRAPPER}} .bar' => 'background: {{VALUE}}',
        ],
      ]
    );

    $this->add_control(
      'width',
      [
        'label' => __( 'Width', ELEMENTOR_MENU42_SLUG ),
        'type' => \Elementor\Controls_Manager::SLIDER,
        'size_units' => [ 'px' ],
        'range' => [
          'px' => [
            'min' => 0,
            'max' => 100,
            'step' => 1,
          ],
        ],
        'default' => [
          'unit' => 'px',
          'size' => 24,
        ],
        'selectors' => [
          '{{WRAPPER}}' => 'width: {{SIZE}}{{UNIT}};',
        ],
      ]
    );

    $this->add_control(
      'height',
      [
        'label' => __( 'Height', ELEMENTOR_MENU42_SLUG ),
        'type' => \Elementor\Controls_Manager::SLIDER,
        'size_units' => [ 'px' ],
        'range' => [
          'px' => [
            'min' => 0,
            'max' => 100,
            'step' => 1,
          ],
        ],
        'default' => [
          'unit' => 'px',
          'size' => 21,
        ],
        'selectors' => [
          '{{WRAPPER}}' => 'height: {{SIZE}}{{UNIT}}',
        ],
      ]
    );

    $this->end_controls_section();
  }

  /**
   * Render widget
   * @inheritdoc \Elementor\Widget_Base::render()
   */
  protected function render(): void {
    $settings = $this->get_settings_for_display();
    $this->render_output($settings);
  }

  /**
   * Render content
   */
  private function render_output(array $settings): void {
    $this->add_render_attribute( '_wrapper', 'class', 'a42-menu-button' );
    $this->add_render_attribute( '_wrapper', 'data-breakpoint', $settings['breakpoint']['size'] );

    echo "<div class='bar top'></div>";
    echo "<div class='bar middle one'></div>";
    echo "<div class='bar middle two'></div>";
    echo "<div class='bar bottom'></div>";
  }
}