
class sticky {

    /**
     * Breakpoint.
     * @type {number}
     */
    breakpoint = 1200;

    /**
     * The element.
     * @type {HTMLElement}
     */
    element;

    /**
     * Admin bar element.
     * @type {HTMLElement | null}
     */
    adminBar;

    /**
     * Clone DIV for height.
     * @type {HTMLElement}
     */
    clone;

    /**
     * Initial Height of the element.
     * @type {number}
     */
    elementHeight = 0;

    /**
     * Initial Height of the element (mobile).
     * @type {number}
     */
    elementHeightMobile = 0;

    /**
     * Admin bar height.
     * @type {number}
     */
    adminBarHeight = 0;

    /**
     * Screen is mobile?
     * @type {boolean}
     */
    isMobile = false;

    /**
     * @type {number}
     * @description The offset of scroll to change transparent to solid.
     */
    scrolledOffsetSolid = 0;

    /**
     * @constructor
     * @param element {HTMLElement} Container element.
     */
    constructor(element) {
        this.element = element;
        this.adminBar = document.querySelector('#wpadminbar');
        this.scrolledOffsetSolid = parseInt(element.getAttribute('transparent-to-solid'));
        const button = this.element.querySelector('.elementor-widget-elementor-menu42-menu-button');
        if (!!button) {
            this.breakpoint = parseInt(button.dataset.breakpoint);
        }
        this.init();
    }

    /**
     * Initialize the class.
     * @returns {void}
     */
    init() {
        this.isMobile = window.innerWidth <= this.breakpoint;
        this.getCssVariableHeight();
        this.addEventListeners();
        this.calculateAdminBarHeight();

        // Create clone
        this.createClone();

        // Force first load
        this.onScroll();
    }

    /**
     * Create a clone box of the element. (only if [transparent-to-solid] is not set
     * @returns {void}
     */
    createClone() {
        this.clone = document.createElement('div');
        this.clone.style.width = '100%';
        this.clone.classList.add('sticky-clone');
        this.element.parentNode.insertBefore(this.clone, this.element);

        this.element.style.position = 'fixed';
        this.setCloneHeight();
    }

    /**
     * Get the CSS variables height --header-menu-height
     * and --header-menu-height-mobile.
     */
    getCssVariableHeight() {
        const style = window.getComputedStyle(this.element);
        this.elementHeight = parseInt(style.getPropertyValue('--header-menu-height')) ?? 100;
        this.elementHeightMobile = parseInt(style.getPropertyValue('--header-menu-height-mobile')) ?? 60;
    }

    /**
     * Calculate the admin bar height.
     * @returns {void}
     */
    calculateAdminBarHeight() {
        this.adminBarHeight = this.adminBar ? this.adminBar.offsetHeight : 0;
    }

    /**
     * Add event listeners.
     * @returns {void}
     */
    addEventListeners() {
        window.addEventListener('scroll', this.onScroll.bind(this));
        window.addEventListener('resize', this.onResize.bind(this));
    }

    /**
     * On scroll event.
     * @returns {void}
     */
    onScroll() {
        this.setPosition();
    }

    /**
     * On resize event.
     * @returns {void}
     */
    onResize() {
        this.isMobile = window.innerWidth <= this.breakpoint;
        this.calculateAdminBarHeight();
        this.setCloneHeight();
        this.setPosition();
    }

    /**
     * Set clone height
     * @returns {void}
     */
    setCloneHeight() {
        if (!this.scrolledOffsetSolid || this.scrolledOffsetSolid === NaN ) {
            this.clone.style.height = this.isMobile ? this.elementHeightMobile + 'px' : this.elementHeight + 'px';
        } else {
            this.clone.style.height = 0;
        }
    }

    /**
     * Set the position of the element.
     * @returns {void}
     */
    setPosition() {
        const clonePosition = this.clone.getBoundingClientRect().top;
        const adminBar = window.innerWidth < 601 ? 0 : this.adminBarHeight;
        const pos = clonePosition >= adminBar ? clonePosition : adminBar;
        this.element.style.top = pos + 'px';
    }
}

document.querySelectorAll('#main-header.sticky-top').forEach(element =>
    new sticky(element)
);