
class Expansion {
    /**
     * Breakpoint.
     * @type {number}
     */
    breakpoint = 1024;

    /**
     * The element.
     * @type {HTMLElement}
     */
    element;

    /**
     * Expended Height of the element.
     * @type {number}
     */
    expendedHeight = 0;

    /**
     * Is expanded?
     * @type {boolean}
     */
    expanded = false;

    /**
     * Parent element.
     * @type {HTMLElement}
     */
    parent;

    /**
     * Expand button - Is .item-title if it's not a link or .a42-submenu-arrow-down.
     * @type {HTMLElement | null}
     */
    expandButton;

    /**
     * Screen is mobile?
     * @type {boolean}
     */
    isMobile = false;

    /**
     * Css transition from the element.
     * @type {string}
     */
    transition = '';

    /**
     * Depth of the element.
     * @type {number}
     */
    depth = 0;

    /**
     * Parent expansion Element
     * @type {Expansion}
     */
    parentExpansion;

    /**
     * @constructor
     * @param element {HTMLElement} Container element.
     */
    constructor(element) {
        this.element = element;
        this.element.Expansion = this;
        this.parent = this.element.closest('.has-children');
        this.expandButton = this.parent.querySelector('.item-title:not(a)') ??
            (this.parent.querySelector('.a42-submenu-arrow-down') ?? this.parent.querySelector('.a42-submenu-arrow-right'));

        this.transition = window.getComputedStyle(this.element).transition;
        const header = this.element.closest('#main-header').querySelector('.elementor-widget-elementor-menu42-menu-button');
        this.depth = parseInt(this.element.dataset.depth);

        this.parentExpansion = this.parent.closest('.expand-mobile')?.Expansion;

        if (!!header) {
            this.breakpoint = parseInt(header.dataset.breakpoint);
        }
        if (this.parent) {
            this.init();
        }
    }

    /**
     * Initialize the class.
     * @returns {void}
     */
    init() {
        this.isMobile = window.innerWidth <= this.breakpoint;
        this.addEventListeners();
        this.isMobile ? this.onResizeToMobile() : this.onResizeToDesktop();
    }

    onResizeToDesktop() {
        this.collapse();
        setTimeout(_ => {
            this.element.style.height = `auto`;
            this.element.style.overflow = 'visible';
        },20)

    }

    onResizeToMobile() {
        this.collapse();
        this.element.style.overflow = 'hidden';
    }

    /**
     * Add event listeners.
     * @returns {void}
     */
    addEventListeners() {
        if (!!this.expandButton) {
            this.expandButton.addEventListener('click', this.toggle.bind(this));
        }

        // Resize
        window.addEventListener('resize', () => {
            if (window.innerWidth <= this.breakpoint && !this.isMobile) {
                this.isMobile = true;
                this.onResizeToMobile();
            } else if (window.innerWidth > this.breakpoint && this.isMobile) {
                this.isMobile = false;
                this.onResizeToDesktop();
            }
        });
    }

    /**
     * Update the element height original value.
     * @param {number} height
     * @returns {void}
     */
    updateHeight(height = 0) {
        const currentHeight = this.element.offsetHeight;
        setTimeout(_ => {
            this.element.style.height = `${currentHeight + height}px`;
        }, 10);

    }

    /**
     * Toggle the element.
     * @param {MouseEvent} event
     * @returns {void}
     */
    toggle(event) {
        if (this.isMobile) {
            event.stopPropagation();
            event.preventDefault();
            this.expanded ? this.collapse() : this.expand();
            this.expanded = !this.expanded;
        }
    }

    /**
     * Get Expanded Height.
     * @returns {Promise<number>}
     */
    async getExpandedHeight() {
        return new Promise(resolve => {
            setTimeout(_ => {
                this.element.style.transition = 'none';
                this.element.style.height = 'auto';
                this.parent.classList.add('expanded');
                const expendedHeight = this.element.offsetHeight
                this.element.style.transition = this.transition;
                this.collapse();
                resolve(expendedHeight);
            }, 10);
        })
    }

    /**
     * Expand the element.
     * @param {number} modifier - Modifier to the height (from Children).
     * @returns {void}
     */
    async expand(modifier = 0) {

        const expendedHeight = await this.getExpandedHeight();

        setTimeout(_ => {
            this.parentExpansion?.updateHeight(expendedHeight);
            this.element.style.height = `${expendedHeight + modifier}px`;
            this.parent.classList.add('expanded');
        }, 10);
    }

    /**
     * Collapse the element.
     * @returns {void}
     */
    collapse() {
        //this.expanded = false;
        this.parentExpansion?.updateHeight(-this.element.offsetHeight);
        this.element.style.height = '0';
        this.parent.classList.remove('expanded');
    }
}

document.querySelectorAll('.expand-mobile').forEach(element =>
    new Expansion(element)
);