<?php
namespace Brand42\Modules\DashboardWidgets;

if ( ! defined( 'ABSPATH' ) ) {
  exit; // Exit if accessed directly.
}

/**
 * Contact Widget
 *
 * @package Brand42\Modules\DashboardWidgets
 */
class PluginUpdates extends DashboardBaseWidget {

  public function __construct() {
    parent::__construct();

    // Add the ajax action to send the mail
    add_action('wp_ajax_send_plugin_update_mail', [$this, 'send_plugin_update_mail']);
  }

  public function get_title(): string {
    return 'Extensions à mettre à jour';
  }

  public function get_name(): string {
    return 'brand42_plugin_updates_widget';
  }

  public function get_context(): string {
    return 'column3';
  }

  protected function render(): void {
    $infos = $this->get_site_infos();
    echo '<div class="flex-column gap-10">';

    if ( empty( $infos['plugin_updates_list'] ) ) {
      echo 'Toutes vos extensions sont à jour.';
    } else {
      echo 'Les extensions suivantes doivent être mises à jour :';
      $this->render_plugin_updates( $infos['plugin_updates_list']);
    }

    $updated_plugins_count = count($infos['plugin_updated_list']);
    if ( $updated_plugins_count > 0 ) {
      echo '<div class="num-plugins flex mg-bottom flex align-center gap-5"><span class="num">' . $updated_plugins_count . '</span> extensions sont déjà à jour</div>';
    }

    // Ask for updates button -> send an automatic mail to the admin
    echo '<div id="ask-update" class="a42-button a42-button--primary">Demander une maintenance</div>';

    echo '</div>';

  }

  private function render_plugin_updates($plugin_updates): void {
    echo '<table class="a42-table">';
    echo '<thead>';
    echo '<tr>';
    echo '<th>Extension</th>';
    echo '<th>Version actuelle</th>';
    echo '<th>Nouvelle version</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody>';
    foreach ( $plugin_updates as $plugin ) {
      echo '<tr>';
      echo '<td>' . $plugin['name'] . '</td>';
      echo '<td>' . $plugin['current_version'] . '</td>';
      echo '<td>' . $plugin['new_version'] . '</td>';
      echo '</tr>';
    }
    echo '</tbody>';
    echo '</table>';




  }

  public function send_plugin_update_mail(): void {
    $infos = $this->get_site_infos();
    $to = 'hebergement@agence42.fr, support@agence42.fr';
    // From is site@domain
    $from = 'site@' . $_SERVER['SERVER_NAME'];

    $subject = '[' . $infos['site_name'] . '] Demande de maintenance';
    $nice_update_list = '<ul>';
    foreach ($infos['plugin_updates_list'] as $plugin) {
      $nice_update_list .= '<li><b>' .$plugin['name'] . '</b> : '. $plugin['current_version'] . ' &#8594; : ' . $plugin['new_version'] . '</li>';
    }
    if ( !empty($infos['wp_update']->updates) && $infos['wp_is_up_to_date'] === false) {
      $nice_update_list .= '<li><b>WordPress</b> : ' . $infos['wp_version'] . ' &#8594; : ' . $infos['wp_update_current'] . '</li>';
    }
    $nice_update_list .= '</ul>';
    $nice_update_list .= '<br>Version PHP : ' . $infos['php_version'];
    $nice_update_list .= '<br>Version WordPress : ' . $infos['wp_version'] . '<i>(' . $infos['wp_is_up_to_date_str'] . ')</i>';

    $message = '<h1>Demande de maintenance</h1>';

    $message .= '<p>Le client pour le site <a href="' . $infos['site_url'] . '">' . $infos['site_name'] . '</a> demande a demandé une mise à jour des extensions.</p>';
    $message .= '<p>Voici la liste des extensions à mettre à jour : </p>';
    $message .= $nice_update_list;

    if (!empty($infos['plugin_updated_list'])) {
      $message .= '<hr>';
      $nice_updated_list = '<ul>';
      foreach ($infos['plugin_updated_list'] as $plugin) {
        $nice_updated_list .= '<li><b>' . $plugin['Name'] . '</b> : ' . $plugin['Version'] . '</li>';
      }
      $nice_updated_list .= '</ul>';
      $message .= '<p>Voici la liste des extensions déjà à jour : </p>';
      $message .= $nice_updated_list;
      $message .= '<hr>';
      $message .= '<small>Cette demande a été envoyée depuis le site ' . $infos['site_name'] . ' (' . $_SERVER['SERVER_NAME'] . ').</small>';
    }

    $headers = [
      'From: ' . $infos['site_name'] . ' <' . $from . '>',
      'Content-Type: text/html; charset=UTF-8'
    ];

    wp_mail($to, $subject, $message, $headers);

    wp_send_json_success( $to);
  }


  private function get_site_infos(): array {
    $infos = [];

    // Site tech infos
    $infos['site_name'] = get_bloginfo('name');
    $infos['site_url'] = get_bloginfo('url');
    $infos['php_version'] = phpversion();
    $infos['wp_version'] = get_bloginfo('version');
    $infos['wp_update'] = get_site_transient( 'update_core' );

    $infos['all_plugins'] = get_plugins();
    $infos['plugins'] = get_site_transient( 'update_plugins' );
    $infos['plugin_updates'] = $infos['plugins']->response;
    $infos['plugin_updates_list'] = [];
    $infos['wp_is_up_to_date'] = false;
    $infos['plugin_updated_list'] = [];

    if (!empty($infos['wp_update']->updates)) {
      $infos['wp_update_current'] = $infos['wp_update']->updates[0]->current;
      $infos['wp_is_up_to_date'] = $infos['wp_update']->updates[0]->current == $infos['wp_version'];
    } else {
      $infos['wp_update_current'] = $infos['wp_version'];
    }
    $infos['wp_is_up_to_date_str'] = $infos['wp_is_up_to_date'] ? 'À jour' : 'En retard';

    foreach ($infos['plugin_updates'] as $index => $plugin) {
      $current_plugin = $infos['all_plugins'][$index];
      $infos['plugin_updates_list'][] = [
        'name' => $current_plugin['Name'],
        'current_version' => $current_plugin['Version'],
        'new_version' => $plugin->new_version
      ];
    }

    foreach ($infos['all_plugins'] as $index => $plugin) {
      // If plugin is updated
      if (empty($infos['plugin_updates'][$index])) {
        $infos['plugin_updated_list'][$index] = $plugin;
      }
    }

    return $infos;
  }
}