<?php
namespace Brand42\Modules\OptionsPage;

if ( ! defined( 'ABSPATH' ) ) {
  exit; // Exit if accessed directly.
}

/**
 * BaseSettings Class
 *
 * @package Brand42\Modules\OptionsPage
 */
abstract class BaseSettings {
  private array $settings = [];
  private static array $sections = [];
  public function __construct() {
    add_action('admin_init', [$this, 'init']);
    add_action('admin_init', [$this, 'wp_register_settings']);
  }
  public function init(): void {
    $this->register_settings();
  }

  public abstract function get_section(): string;

  public abstract function get_title(): string;

  public abstract function register_settings(): void;

  /**
   * Add a setting
   * @param $setting [
   *  'option' => 'option_unique_name',
   *  'type' => 'text', (Will use the method render_field_type_text)
   *  'label' => 'Option Label',
   *  'default' => 'Default Value',
   * ]
   * @return void
   */
  public function add_setting($setting): void {
    $this->settings[] = $setting;
  }

  public function wp_register_settings(): void {
    $section = $this->get_section();

    // If section isn't in array, add it
    if (!in_array($section, self::$sections)) {
      self::$sections[] = $section;
      add_settings_section(
        $section, $this->get_title(),
        [$this, 'section_callback'],
        'brand42_options');
    }

    foreach ($this->settings as $setting) {
      register_setting('brand42_options', $setting['option']);
      add_settings_field(
        $setting['option'],
        $setting['label'],
        [$this, 'render_field_type_' . $setting['type']],
        'brand42_options',
        $section,
        $setting);

      // IF default value is set, set it
      if (!get_option($setting['option']) && isset($setting['default'])) {
        update_option($setting['option'], $setting['default']);
      }
    }
  }

  /**
   * Section callback (override this method in child class)
   * @param $section
   * @return void
   */
  public function section_callback($section): void {}

  public function render_field_type_text($args): void {
    $value = get_option($args['option']);
    $field = '<input type="text" name="' . $args['option'] . '" value="' . $value . '">';
    $this->field_layout($args, $field);
  }

  public function render_field_type_checkbox($args): void {
    $value = get_option($args['option']);
    $field = '<input type="checkbox" name="' . $args['option'] . '" ' . checked(1, $value, false) . '>';
    $this->field_layout($args, $field);
  }

  public function render_field_type_checkboxes($args): void {
    $value = get_option($args['option']);
    if (!is_array($value)) {
      $value = [];
    }
    $field = '<div class="checkbox-options flex flex-wrap gap-10">';
    foreach ($args['options'] as $key => $label) {
      $field .= '<div class="checkbox-option flex gap-5 align-center">';
      $field .= '<input type="checkbox" name="' . $args['option'] . '[]" value="' . $key . '" ' . checked(true, in_array($key, $value), false) . '>' . $label;
      $field .= '</div>';
    }
    $field .= '</div>';
    $this->field_layout($args, $field);
  }

  public function render_field_type_select($args): void {
    $value = get_option($args['option']);
    $field = '<select name="' . $args['option'] . '">';
    foreach ($args['options'] as $key => $label) {
      $field .= '<option value="' . $key . '" ' . selected($key, $value, false) . '>' . $label . '</option>';
    }
    $field .= '</select>';
    $this->field_layout($args, $field);
  }

  public function render_field_type_radio($args): void {
    $value = get_option($args['option']);
    $field = '';
    foreach ($args['options'] as $key => $label) {
      $field .= '<input type="radio" name="' . $args['option'] . '" value="' . $key . '" ' . checked($key, $value, false) . '>' . $label . '<br>';
    }
    $this->field_layout($args, $field);
  }

  public function render_field_type_switch($args): void {
    $value = get_option($args['option']);
    $field = '<label class="switch"><input type="checkbox" name="' . $args['option'] . '" ' . checked('on', $value, false) . '><span class="slider round"></span></label>';
    $this->field_layout($args, $field);
  }

  public function render_field_type_textarea($args): void {
    $value = get_option($args['option']);
    $field = '<textarea name="' . $args['option'] . '">' . $value . '</textarea>';
    $this->field_layout($args, $field);
  }

  public function render_field_type_number($args): void {
    $value = get_option($args['option']);
    $field = '<input type="number" name="' . $args['option'] . '" value="' . $value . '">';
    $this->field_layout($args, $field);
  }

  public function render_field_type_email($args): void {
    $value = get_option($args['option']);
    $field = '<input type="email" name="' . $args['option'] . '" value="' . $value . '">';
    $this->field_layout($args, $field);
  }

  private function field_layout($args, $field): void {
    echo '<div class="b42-option_field flex-column gap-10 b42-option_field-'.$args['type'].'">';
    if (!empty($args['description'])) {
      echo '<div class="option_desc">' . $args['description'] . '</div>';
    }
    echo $field;
    echo '</div>';
  }


}