
class Form42 {

    /**
     * Element Component
     * @type {HTMLElement}
     */
    element;

    /**
     * Element form
     * @type {HTMLFormElement}
     */
    formElement;

    /**
     * Is carousel step form
     * @type {boolean}
     */
    isCarouselType = false;

    /**
     * Is Elementor42 Validation
     * @type {boolean}
     */
    instantValidation = false;

    /**
     * @constructor
     * @param element {HTMLElement} The form element
     */
    constructor(element) {
        if (!element) return;
        this.element = element;
        this.formElement = element.querySelector('form');
        if (!this.formElement) return;
        this.formElement.Form42 = this;
        if (!this.formElement) return;
        this.isCarouselType = this.element.classList.contains('step-carousel');
        this.instantValidation = this.element.classList.contains('elementor42-validation');

        const self = this;
        this.init();


    }

    init() {
        if (this.isCarouselType) {
            this.initCarousel();
        }

        if (this.instantValidation) {
            this.FormValidation = new FormValidation(this.formElement, this.isCarouselType);
        }
    }

    initCarousel() {
        this.formElement.addEventListener('click', (event) => {
            if (event.target.tagName === 'BUTTON') {
                if (event.target.classList.contains('e-form__buttons__wrapper__button-previous') || event.target.classList.contains('e-form__buttons__wrapper__button-next')) {
                    this.onNextPrevClick(event);
                }
            }
        });

        this.formElement.addEventListener('submit', this.onSubmit.bind(this));
        setTimeout(() => {
            this.formElement.classList.add('loaded');
        }, 200);
    }

    onNextPrevClick(event) {
        if (this.isCarouselType) {
            // Scroll top of the form
            const headerHeight = document.querySelector('header#main-header')?.offsetHeight ?? 0;
            window.scrollTo({
                // 50 is the margin-top of the form
                top: this.element.getBoundingClientRect().top + window.scrollY - headerHeight - 50,
                behavior: 'smooth'
            });
        }
    }

    onSubmit(event) {
        this.formElement.classList.add('submitting');
        setTimeout(() => {
            this.formElement.classList.remove('submitting');
        }, 200);

    }


}


class FormValidation {
    /**
     * Element form
     * @type {HTMLFormElement}
     */
    formElement;

    /**
     * Step buttons (next)
     * @type {HTMLButtonElement[]}
     */
    nextButtons = [];

    /**
     * Submit button
     * @type {HTMLButtonElement}
     */
    submitButton;

    /**
     * Inputs
     * @typedef {{
     *     input: HTMLInputElement,
     *     valid: boolean
     *     nextButton: HTMLButtonElement
     *     getValidity: function
     *     error: string
     * }[]} inputs - Array of inputs
     */
    inputs = [];

    buttonsInited = false;

    /**
     * Errors
     * @param formElement
     * @param isCarouselType
     */
    constructor(formElement, isCarouselType = false) {
        this.formElement = formElement;
        this.isCarouselType = isCarouselType;

        this.submitButton = this.formElement.querySelector('button[type="submit"]');
        this.init();
    }

    init() {
        //this.checkInput();
        //this.getNextButtons();
        //this.manageSubmitState();



        if (this.isCarouselType) {
            this.getNextButtons(_ => {
                this.getAllInputs();
                this.nextButtons.forEach(button => {
                    this.manageButton(button);
                    this.manageButton(this.submitButton);
                    this.checkButtonValidity(button);
                    this.checkButtonValidity(this.submitButton);
                });
            });
        } else {
            this.manageButton(this.submitButton);
            this.getAllInputs(this.submitButton);
            this.checkButtonValidity(this.submitButton);
        }
    }

    showCurrentStepErrors(button) {
        const inputsForButton = this.getInputsForBtn(button);
        inputsForButton.forEach(input => {
            if (!input.valid) {
                input.input.dispatchEvent(new Event('showError'));
            }
        });
    }

    /**
     * Intercept button next and submit
     */
    manageButton(button) {
        const self = this;
        // Scope is step (if carousel) or the form if not carousel
        const scope = this.isCarouselType ? button.closest('.elementor-field-type-step') : this.formElement;
        scope?.addEventListener('invalid', (function(){
            return function(e) {
                e.preventDefault();
                self.showCurrentStepErrors(button);
            };
        })(), true);

        button.setAttribute('data-valid', 'false');
        button.addEventListener('click', event => {
            this.checkButtonValidity(button);
            if (button.getAttribute('data-valid') === 'false') {
                // Show errors
                this.showCurrentStepErrors(button);

                event.preventDefault();
                event.stopPropagation();
                button.setAttribute('disabled', 'disabled');
            }
        }, true);
    }


    /**
     * Check input validity
     */
    inputValidation = [
        {
            selector: '.a42-validation',
            validate: input => {
                if (input.getValidity !== undefined) {
                    return input?.getValidity()
                } else {
                    return input.value !== '';
                }
            }
        },
        {
            selector: `
                .elementor-field-type-text input[type="text"][required="required"],
                .elementor-field-type-tel input[type="tel"][required="required"],
                .elementor-field-type-url input[type="url"][required="required"]
                `,
            validate: input => {
                return input.value !== '';
            }
        },
        {
            selector: '.elementor-field-type-select select[required="required"]',
            validate: input => {
                return input.value !== '';
            }
        },
        {
            selector: '.elementor-field-type-email input[type="email"]',
            validate: input => {
                const isFieldRequired = input.getAttribute('required') != null;
                const isValidEmail = input.value.match(/^([\w.%+-]+)@([\w-]+\.)+([\w]{2,})$/i);
                if (isFieldRequired) {
                    return isValidEmail && input.value !== '';
                } else {
                    return isValidEmail || input.value === '';
                }
            }
        }
    ];

    checkButtonValidity(button) {
        const inputsForButton = this.getInputsForBtn(button);
        let valid = true;
        inputsForButton.forEach(i => {
            //i.input.dispatchEvent(new Event('buttonCheckErrors'));
            if (!i.getValidity(i.input)) {
                valid = false;
            }
        })

        if (valid) {
            button.removeAttribute('disabled');
            button.setAttribute('data-valid', 'true');
        } else {
            button.setAttribute('data-valid', 'false');
        }
    }

    /**
     * Get all inputs of the form
     */
    getAllInputs() {
        if (this.buttonsInited) return;
        this.buttonsInited = true;

        this.inputValidation.forEach(validation => {
            this.formElement.querySelectorAll(validation.selector).forEach(input => {
                this.inputs.push({
                    input: input,
                    valid: validation.validate(input),
                    getValidity: validation.validate,
                    nextButton: input.closest('.elementor-field-type-step')?.querySelector('.e-form__buttons__wrapper__button-next') ?? this.submitButton,
                });
            });
        });

        this.inputs.forEach(input => {
            input.input.addEventListener('input', () => {
                input.valid = input.getValidity(input.input);
                this.checkButtonValidity(input.nextButton);
            });

            input.input.addEventListener('validate', () => {
                input.valid = input.getValidity(input.input);
                this.checkButtonValidity(input.nextButton);
                if (!input.valid) {
                    input.input.dispatchEvent(new Event('showError'));
                } else {
                    input.input.dispatchEvent(new Event('removeError'));
                }
            });
            input.input.addEventListener('showError', () => {
                this.showInputError(input.input, input.input.error);
            });
            input.input.addEventListener('removeError', () => {
                this.removeInputError(input.input);
            });
        });
    }

    getInputsForBtn(button) {
        return this.inputs.filter(i => i.nextButton === button);
    }

    showInputError(input, message) {
        const currentMessage = input.closest('.elementor-field-group')?.querySelector('.error-message');
        if (!!currentMessage) {
            if (currentMessage.innerHTML === message) {
                return;
            } else {
                currentMessage.remove();
            }
        }
        if (message == null) {
            if (!!currentMessage) {
                currentMessage.remove();
            }
            return;
        }

        const error = document.createElement('span');
        error.classList.add('error-message', 'elementor-message', 'elementor-message-danger', 'elementor-form-help-inline');
        error.innerHTML = message;
        const wrapper = input.closest('.elementor-field-group') ?? input;
        wrapper.classList.add('error');
        wrapper.append(error);
    }

    removeInputError(input) {
        const wrapper = input.closest('.elementor-field-group') ?? input;
        const currentMessage = wrapper?.querySelector('.error-message');
        if (currentMessage) {
            currentMessage.remove();
        }

        wrapper.classList.remove('error');
    }


    /**
     * For step forms, next and previous buttons are generated by Elementor via Javascript,
     * so we need to observe the mutation of the form to get the next and previous buttons.
     */
    getNextButtons(fn) {
        const observer = new MutationObserver(mutations => {
            mutations.forEach(mutation => {
                mutation.addedNodes.forEach(node => {
                    if (node.classList.contains('e-form__buttons')) {
                        const button = node.querySelector('.e-form__buttons__wrapper__button-next');
                        if (button) {
                            this.nextButtons.push(button);
                            fn();
                        }
                    }
                });
            });
        });

        observer.observe(this.formElement, {childList: true, subtree: true});
    }
}

function applyA42DynamicData(data) {
    document.querySelectorAll('.a42-form-action[data-field-id]').forEach(element => {
        const fieldId = element.getAttribute('data-field-id');
        if (data[fieldId] != null) {
            // Replace the span by text
            const before = element.getAttribute('data-before') ?? '';
            const after = element.getAttribute('data-after') ?? '';
            element.innerHTML = before + data[fieldId] + after;
        } else if (element.getAttribute('data-fallback') != null) {
            element.innerHTML = element.getAttribute('data-fallback');
        } else {
            element.innerHTML = '';
        }
    });
}

window.addEventListener('elementor/frontend/init', () => {
    elementorFrontend.hooks.addAction( 'frontend/element_ready/widget', function($scope) {
        $scope.find('.elementor-widget.elementor-widget-form').each((index, element) => {
            const form = element.querySelector('form');
            if (form != null && form.Form42) return;
            new Form42(element);
        });
    });
    elementorFrontend.elements.$document.on('elementor/popup/show', function ($scope) {
        $scope.target.querySelectorAll('.elementor-widget.elementor-widget-form').forEach( element => {
            const form = element.querySelector('form');
            if (form != null &&form.Form42) return;

            // Set timeout because sometimes the form fields are not ready
            setTimeout(() => {
                new Form42(element);
            }, 100);
        });
    });

    // Wait for form submit success
    elementorFrontend.elements.$document.on('submit_success', function (event, response) {
        // Manage a42 Popup form action
        if (response?.data?.a42_popup?.action === 'open' && response?.data?.a42_popup?.id != null) {
            elementorProFrontend.modules.popup.closePopup({}, event);
            const popup = elementorFrontend.documentsManager.documents[response.data.a42_popup.id];

            if (popup != null) {
                popup.showModal();
            }

            if (response.data?.a42_data != null) {
                applyA42DynamicData(response.data.a42_data);
            }
        }
    });
});
