
class LoopFilter42 {
    /**
     * Element Widget
     * @type {Element}
     */
    element;

    /**
     * Buttons
     * @type {NodeListOf<Element>}
     */
    buttons;

    /**
     * Filters
     * @type string[]
     */
    filters = [];

    /**
     * Loop42
     * @type {Loop42}
     */
    Loop42;

    /**
     * is Expandable?
     * @type {boolean}
     */
    isExpandable = false;

    /**
     * is Multiple selection?
     * @type {boolean}
     */
    isMultiple = false;

    /**
     * is the filter42 scrollable?
     * @type {boolean}
     */
    isScrollable = false;

    /**
     * AllButton
     * @type {Element | undefined}
     */
    allButton;

    /**
     * Active filter buttons
     * @type {Element[]}
     */
    activeButtons= [];

    get activeFilters() {
        return this.activeButtons.map(button => button.parentElement.getAttribute('data-term-id'));
    }

    /**
     * @constructor
     * @param {Element} element
     */
    constructor(element) {
        this.element = element;
        this.Loop42 = document.querySelector('.elementor-widget-a42-loop42')?.Loop42;
        if (this.element == null || this.Loop42 == null) return;

        this.allButton = this.element.querySelector('[data-term-id="-1"] > .item');

        this.isExpandable = this.element.querySelector('[data-expandable]') != null;
        this.isMultiple = this.element.querySelector('[data-multiple]') != null;
        this.isScrollable = this.element.querySelector('[data-scrollable]') != null;
        this.init();
    }

    /**
     * Init
     */
    init() {
        this.buttons = this.element.querySelectorAll('.a42-loop-filter42--item');
        this.listenEvents();
        this.manageScroll();
    }

    /**
     * Listen events
     */
    listenEvents() {
        this.buttons.forEach(button => {
            /** @event click */
            button.addEventListener('click', event => {
                this.manageActive(button);
                this.manageExpand(button);
                this.fetchData();
            })
        });
    }

    fetchData() {
        const args = !this.filters.includes('-1') ? {q_include_term_ids: this.activeFilters} : {};
        this.Loop42.query(args);
    }

    manageScroll() {
        if (!this.isScrollable) return;
        const slider = this.element.querySelector('.a42-loop-filter42');

        // Scroll on drag
        let isDown = false;
        let startX;
        let scrollLeft;

        slider.addEventListener('mouseenter', () => {
            // Check if scroll is needed
            const scrollWidth = slider.scrollWidth;
            const clientWidth = slider.clientWidth;
            if (scrollWidth <= clientWidth){
                slider.classList.remove('scrollable');
            } else {
                slider.classList.add('scrollable');
            }

        });

        // Scroll on grab
        slider.addEventListener('mousedown', e => {
            slider.classList.add('grabbing');
            isDown = true;
            slider.classList.add('active');
            startX = e.pageX - slider.offsetLeft;
            scrollLeft = slider.scrollLeft;
        });
        document.addEventListener('mouseup', event => {
            slider.classList.remove('grabbing');
            isDown = false;
            slider.classList.remove('active');
        });

        document.addEventListener('mousemove', e => {
            if (!isDown) return;
            e.preventDefault();
            e.stopPropagation();
            const x = e.pageX - slider.offsetLeft;
            const walk = (x - startX); //scroll-fast
            slider.scrollLeft = scrollLeft - walk;

        })
    }
    /**
     * Active
     * @param button {Element}
     */
    manageActive(button) {
        if (button == null) return;
        this.isMultiple ? this._manageMultipleActive(button) : this._manageSingleActive(button);
    }

    /**
     * Manage Single active
     * @param button {Element}
     */
    _manageSingleActive(button) {
        const filter = button.parentElement.getAttribute('data-term-id');
        if (this.activeFilters.includes(filter) && filter !== '-1') {
            // Remove filter
            this.activeButtons = [];
            this.buttons.forEach(button => {
                button.classList.remove('active')
                button.parentElement.classList.remove('active');
            });
        } else {
            // Add filter
            this.activeButtons = [button];
            this.buttons.forEach(button => button.classList.remove('active'));
            button.classList.add('active');
            button.parentElement.classList.add('active');
        }
    }

    /**
     * Manage Multiple active
     * @param button {Element}
     */
    _manageMultipleActive(button) {
        // Manage all button
        if (this.allButton != null && button === this.allButton) {
            button.classList.add('active');
            button.parentElement.classList.add('active');
            this.buttons.forEach(button => button.classList.remove('active'));
            this.activeButtons = [button];
        } else if (this.allButton != null) {
            this.allButton.classList.remove('active');
        }

        // Manage filter buttons
        let activating = false;
        if (button.classList.contains('active')) {
            button.classList.remove('active');
            button.parentElement.classList.remove('active');
            activating = false;
        } else {
            button.classList.add('active');
            button.parentElement.classList.add('active');
            activating = true;
        }
        const children = this.findAncestor(button, 'a42-loop-filter42--item-wrapper')?.querySelectorAll('.items-wrapper .item');

        // Item has children ?
        if (children != null && children.length !== 0) {
            if (activating) {
                /** @Activating **/
                children.forEach(child => child.classList.add('active'));

                // Check if all children are active (and activate parent)
                const parentWrapper = this.findAncestor(button.parentElement, 'a42-loop-filter42--item-wrapper')?.querySelector('.item');
                if (parentWrapper != null) {
                    const allActives = this.__allChildrenActive(parentWrapper);
                    if (allActives) {
                        parentWrapper.classList.add('active');
                    }
                }

            } else {
                /** @Deactivating **/
                children.forEach(child => child.classList.remove('active'));

                // Check if any parent is active (and deactivate it)
                let parentWrapper = this.findAncestor(button, 'items-wrapper');
                while (parentWrapper != null && parentWrapper.getAttribute('data-level') !== '0') {
                    const parentButton = this.findAncestor(parentWrapper, 'a42-loop-filter42--item-wrapper')
                        ?.querySelector('.a42-loop-filter42--item');

                    if (parentButton != null) {
                        parentButton.classList.remove('active');
                    }
                    parentWrapper = this.findAncestor(parentWrapper, 'items-wrapper');
                }
            }
        }

        // Check if we reactive allButton (if all buttons are inactive)
        const activeButtons = this.element.querySelectorAll('.a42-loop-filter42--item.active');
        if (activeButtons.length === 0 && this.allButton != null) {
            this.allButton.classList.add('active');
        }
        /**
         * Calculate active filters (if item is active and all its children are active, then
         * add only the parent item to the filters)
         */
        this.activeButtons = [];

        this.element.querySelectorAll('[data-level="0"] > .a42-loop-filter42--item-wrapper').forEach(scope => {
            this._calculateFilters(scope);
        });
    }

    /**
     * Calculate filters (from multiple active buttons)
     * Calculate active filters (if item is active and all its children are active, then
     * add only the parent item to the filters)
     * @param scope {Element}
     */
    _calculateFilters(scope) {
        if (scope == null) return;
        const item = this.queryDirectChilds(scope, 'item')[0];
        if (item == null) return;
        const itemHasChildren = scope.querySelector('.items-wrapper') != null;
        const isActive = item.classList.contains('active');
        if (isActive && !itemHasChildren) {
            this.activeButtons.push(item);
        } else if (isActive && this.__allChildrenActive(item)) {
            // Check if item is active, and if all its children items are active
            this.activeButtons.push(item);
        } else {
            // Re run this function for the children
            const subScopeWrapper = scope.querySelector('.items-wrapper');
            this.queryDirectChilds(subScopeWrapper, 'a42-loop-filter42--item-wrapper').forEach(subscope => {
                this._calculateFilters(subscope);
            })
        }

    }

    /**
     * Check if all children of an element are active
     */
    __allChildrenActive(element) {
        const children = element.parentElement.querySelector('.items-wrapper')?.querySelectorAll('.item');
        if (children == null) return false;
        const activeChildren = Array.from(children).filter(child => child.classList.contains('active'));
        return activeChildren.length === children.length;
    }

    /**
     * Expand
     * @param button {Element}
     */
    manageExpand(button) {
        if (button == null) return;
        if (this.isExpandable) {
            // Element is sibling of button '.items-wrapper';
            const parent = button.parentElement;
            const el = parent.querySelector('.a42-loop-filter42--item ~ .items-wrapper');
            if (el != null) {
                el?.classList.contains('open') ? this.closeExpand(el) : this.openExpand(el);
            } else if (parent.getAttribute('data-term-id') === '-1') {
                // Element is parent of button '.items-wrapper';
                const el = parent.parentElement.querySelector('.items-wrapper');
                el?.classList.contains('open') ? this.closeExpand(el) : this.openExpand(el);
            }

        }
    }

    /**
     * Open expand
     * @param element {Element} - Element to open (.items-wrapper)
     * @param fromHeight {number} - Height from where to start the animation
     */
    openExpand(element, fromHeight = 0) {
        if (element == null) return;
        // close others expanded
        const expanded = this.element.querySelectorAll('.items-wrapper.open');
        expanded.forEach(el => this.closeExpand(el));

        element.classList.add('open');
        element.style.height = 'auto';

        const height = element.clientHeight + 'px';
        element.style.height = fromHeight + 'px';
        const parent = this.findAncestor(element, 'items-wrapper');
        if (parent != null && parent.getAttribute('data-level') !== '0') {
            parent.style.height = 'auto';
        }
        setTimeout(_ => {
            element.style.height = height;
        }, 0);
    }

    /**
     * Close expand
     * @param element {Element} - Element to close (.items-wrapper)
     */
    closeExpand(element) {
        if (element == null) return;
        if (element.parentElement.querySelector('.active') != null) return;
        element.style.height = element.clientHeight + 'px';
        setTimeout(_ => {
            element.style.height = '0px';
        }, 0);
        element.addEventListener('transitionend', _ => {
            element.classList.remove('open');
        }, {once: true});
    }

    /**
     * Get Ancestor
     */
    findAncestor(el, cls) {
        let current = el.parentElement;
        while (current) {
            if (current.classList.contains(cls)) {
                return current;
            }
            current = current.parentElement;
        }
        return undefined;
    }

    /**
     * Query Direct child with class
     */
    queryDirectChilds(el, cls) {
        if (el == null) return [];
        return Array.from(el.children).filter(child => child.classList.contains(cls));
    }
}

document.querySelectorAll('.elementor-widget-a42-loop-filter42').forEach(element => {
    new LoopFilter42(element);
});